from fastapi import HTTPException, Query
from typing import Dict, Any
from urllib.parse import quote
from datetime import datetime
from bs4 import BeautifulSoup
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
import logging
import re

from scraper_utils import parse_products_with_browser
from browser_utils import extract_photos_with_browser, analyze_page_structure, create_browser

logger = logging.getLogger(__name__)

BASE_URL = "https://febest.com.au"

def search_products(query: str = Query(..., description="Search query")) -> Dict[str, Any]:
    """
    Search for products by query
    """
    try:
        # Correct search URL format for febest.com.au (Magento-based)
        search_url = f"{BASE_URL}/catalogsearch/result/index/?q={quote(query)}"
        logger.info(f"Searching for products with query: {query} at URL: {search_url}")
        
        products = parse_products_with_browser(search_url)
        
        return {
            "success": True,
            "query": query,
            "total_products": len(products),
            "products": products,
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Error in search_products: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Search failed: {str(e)}")

def shop_by_mark(mark: str = Query(..., description="Car mark/brand")) -> Dict[str, Any]:
    """
    Get products by car mark/brand
    """
    try:
        mark_lower = mark.lower()
        
        # Try different URL formats for categories - using correct febest.com.au format
        url_formats = [
            f"{BASE_URL}/shop-parts.html?find={mark_lower}",
            f"{BASE_URL}/shop-parts.html?find={mark}",  # Try original case too
            f"{BASE_URL}/shop-parts.html?find={mark.upper()}",  # Try uppercase
        ]
        
        products = []
        successful_url = None
        
        # Try each URL format
        for test_url in url_formats:
            logger.info(f"Trying URL format: {test_url}")
            try:
                test_products = parse_products_with_browser(test_url)
                if test_products:  # If we found products, use this URL
                    products = test_products
                    successful_url = test_url
                    logger.info(f"✅ Found {len(products)} products at: {test_url}")
                    break
                else:
                    logger.info(f"⚠️ No products found at: {test_url}")
            except Exception as url_error:
                logger.warning(f"❌ Failed to load: {test_url} - {str(url_error)}")
                continue
        
        # If no category URL worked, try search as fallback
        if not products:
            logger.info(f"No category pages worked, trying search for: {mark}")
            search_url = f"{BASE_URL}/catalogsearch/result/index/?q={quote(mark)}"
            try:
                products = parse_products_with_browser(search_url)
                if products:
                    successful_url = search_url
                    logger.info(f"✅ Found {len(products)} products via search")
                else:
                    logger.warning(f"⚠️ No products found via search either")
            except Exception as search_error:
                logger.error(f"❌ Search also failed: {str(search_error)}")
        
        return {
            "success": True,
            "mark": mark,
            "total_products": len(products),
            "products": products,
            "successful_url": successful_url,
            "method": "category" if successful_url and "catalogsearch" not in successful_url else "search",
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Error in shop_by_mark: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Failed to fetch products for mark {mark}: {str(e)}")

def get_product_details(url: str = Query(..., description="Product URL or slug")) -> Dict[str, Any]:
    """
    Get detailed information about a specific product
    """
    try:
        # Build proper product URL
        if not url.startswith('http'):
            if url.startswith('/'):
                product_url = f"{BASE_URL}{url}"
            else:
                # Add .html extension if not present for Magento products
                if not url.endswith('.html'):
                    url = f"{url}.html"
                product_url = f"{BASE_URL}/{url}"
        else:
            product_url = url
            
        logger.info(f"Fetching product details from: {product_url}")
        
        browser = None
        try:
            browser = create_browser()
            if not browser:
                raise Exception("Failed to create browser")
                
            browser.get(product_url)
            
            # Wait for page to load with multiple conditions
            WebDriverWait(browser, 15).until(
                lambda driver: driver.execute_script("return document.readyState") == "complete"
            )
            
            # Additional wait for dynamic content
            try:
                WebDriverWait(browser, 10).until(
                    EC.any_of(
                        EC.presence_of_element_located((By.CSS_SELECTOR, "h1")),
                        EC.presence_of_element_located((By.CSS_SELECTOR, ".product-title")),
                        EC.presence_of_element_located((By.CSS_SELECTOR, ".page-title"))
                    )
                )
            except:
                pass  # Continue even if specific elements not found
            
            soup = BeautifulSoup(browser.page_source, 'html.parser')
            
            # Check for 404 or error pages
            page_text = soup.get_text().lower()
            if any(error in page_text for error in ["404", "not found", "page not found"]):
                return {
                    "success": False,
                    "error": "Product not found (404)",
                    "url": product_url
                }
            
            name = ""
            price = ""
            description = ""
            images = []
            
            # Enhanced name selectors for Magento
            name_selectors = [
                'h1.page-title span',
                'h1.page-title',
                'h1.product-title',
                'h1.product-name', 
                '.product-title',
                '.product-name',
                '.product-item-name',
                'h1',
                '.page-title',
                '[data-ui-id="page-title-wrapper"]'
            ]
            
            for selector in name_selectors:
                name_elem = soup.select_one(selector)
                if name_elem:
                    name = name_elem.get_text(strip=True)
                    if name:  # Only break if we got actual text
                        break
            
            # Enhanced price selectors for Magento
            price_selectors = [
                '.price-box .price',
                '.price-wrapper .price',
                '.regular-price .price',
                '.special-price .price',
                '.price-final_price .price',
                '.price',
                '.product-price',
                '.current-price',
                '[class*="price"]',
                '[data-price-type="finalPrice"]'
            ]
            
            for selector in price_selectors:
                price_elem = soup.select_one(selector)
                if price_elem:
                    price_text = price_elem.get_text(strip=True)
                    # Clean price text and extract numeric value
                    price_clean = re.sub(r'[^\d.,\$€£¥₽]', '', price_text)
                    price_match = re.search(r'[\$€£¥₽]?[\d,]+\.?\d*', price_clean)
                    if price_match:
                        price = price_match.group()
                        break
            
            # Enhanced description selectors for Magento
            desc_selectors = [
                '.product.attribute.description .value',
                '.product-info-main .product.attribute.description',
                '.product-description',
                '.product-summary',
                '.description',
                '.summary',
                '.product-details',
                '.product-info',
                '.product.info.detailed',
                '[data-role="content"]'
            ]
            
            for selector in desc_selectors:
                desc_elem = soup.select_one(selector)
                if desc_elem:
                    description = desc_elem.get_text(strip=True)
                    if description and len(description) > 10:  # Only use substantial descriptions
                        break
            
            # Extract product images
            image_selectors = [
                '.product.media img',
                '.product-image-main img',
                '.gallery-image img',
                '.product-gallery img',
                '.fotorama img',
                '.product-image img'
            ]
            
            for selector in image_selectors:
                img_elements = soup.select(selector)
                for img in img_elements:
                    src = img.get('src') or img.get('data-src')
                    if src and src not in images:
                        if src.startswith('/'):
                            src = f"{BASE_URL}{src}"
                        images.append(src)
                if images:
                    break
            
            # Try to extract photos using the existing function
            try:
                photos = extract_photos_with_browser(product_url)
                if photos:
                    images.extend([img for img in photos if img not in images])
            except Exception as e:
                logger.warning(f"Failed to extract photos: {e}")
            
            return {
                "success": True,
                "product": {
                    "name": name or "Product name not found",
                    "url": product_url,
                    "price": price or "Price not found",
                    "description": description or "Description not available",
                    "images": images[:10],  # Limit to first 10 images
                    "url": product_url
                },
                "timestamp": datetime.now().isoformat()
            }
            
        finally:
            if browser:
                browser.quit()
                
    except Exception as e:
        logger.error(f"Error in get_product_details: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Failed to fetch product details: {str(e)}")

def analyze_structure_endpoint(url: str = Query(..., description="URL to analyze")) -> Dict[str, Any]:
    """
    Analyze page structure to understand selectors
    """
    try:
        logger.info(f"Analyzing page structure for: {url}")
        structure = analyze_page_structure(url)
        
        return {
            "success": True,
            "url": url,
            "structure": structure,
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Error in analyze_structure: {str(e)}")
        raise HTTPException(status_code=500, detail=f"Failed to analyze structure: {str(e)}")

def health_check():
    """
    Health check endpoint
    """
    return {
        "status": "healthy",
        "timestamp": datetime.now().isoformat(),
        "version": "2.0.0"
    }