#!/usr/bin/env python3
"""
Production FastAPI application for cheapmotorparts.com
Optimized for shared hosting deployment
"""
import os
import sys
import logging
from pathlib import Path

# Add current directory to Python path
current_dir = Path(__file__).parent
sys.path.insert(0, str(current_dir))

from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse

# Import our routes
try:
    from api_routes import (
        search_products,
        shop_by_mark,
        get_product_details,
        analyze_structure_endpoint,
        health_check
    )
except ImportError as e:
    logging.error(f"Failed to import routes: {e}")
    # Create dummy functions for fallback
    def search_products(*args, **kwargs):
        return {"error": "Import failed", "message": str(e)}
    def shop_by_mark(*args, **kwargs):
        return {"error": "Import failed", "message": str(e)}
    def get_product_details(*args, **kwargs):
        return {"error": "Import failed", "message": str(e)}
    def analyze_structure_endpoint(*args, **kwargs):
        return {"error": "Import failed", "message": str(e)}
    def health_check(*args, **kwargs):
        return {"status": "error", "message": "Import failed"}

# Configure logging for production
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('app.log'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)

# Create FastAPI app
app = FastAPI(
    title="CheapMotorParts API",
    description="Auto parts scraping API for cheapmotorparts.com",
    version="1.0.0",
    docs_url="/docs",
    redoc_url="/redoc"
)

# Configure CORS for production
app.add_middleware(
    CORSMiddleware,
    allow_origins=[
        "https://cheapmotorparts.com",
        "https://www.cheapmotorparts.com",
        "http://cheapmotorparts.com",
        "http://www.cheapmotorparts.com"
    ],
    allow_credentials=True,
    allow_methods=["GET", "POST", "PUT", "DELETE"],
    allow_headers=["*"],
)

# Health check endpoint
@app.get("/")
async def root():
    """Root endpoint"""
    return {
        "message": "CheapMotorParts API is running",
        "version": "1.0.0",
        "status": "healthy"
    }

# Register API routes
app.get("/health")(health_check)
app.get("/search")(search_products)
app.get("/shop-by-mark")(shop_by_mark)
app.get("/product-details")(get_product_details)
app.get("/analyze-structure")(analyze_structure_endpoint)

# Error handler
@app.exception_handler(Exception)
async def global_exception_handler(request, exc):
    logger.error(f"Global exception: {exc}")
    return JSONResponse(
        status_code=500,
        content={"error": "Internal server error", "message": str(exc)}
    )

# For WSGI deployment
application = app

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)