from selenium import webdriver
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import TimeoutException, NoSuchElementException
from webdriver_manager.chrome import ChromeDriverManager
from bs4 import BeautifulSoup
from typing import List, Dict, Any
from urllib.parse import urljoin
import logging
import time

logger = logging.getLogger(__name__)

BASE_URL = "https://febest.com.au"

def create_browser():
    """
    Create and configure Chrome browser instance for scraping
    """
    chrome_options = Options()
    chrome_options.add_argument("--headless")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--disable-dev-shm-usage")
    chrome_options.add_argument("--disable-gpu")
    chrome_options.add_argument("--window-size=1920,1080")
    chrome_options.add_argument("--user-agent=Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36")
    
    try:
        driver = webdriver.Chrome(
            service=webdriver.chrome.service.Service(ChromeDriverManager().install()),
            options=chrome_options
        )
        return driver
    except Exception as e:
        logger.error(f"Failed to create browser: {str(e)}")
        return None

def extract_photos_with_browser(url: str) -> List[str]:
    """
    Extract photos from product page using browser automation
    """
    photos = []
    driver = None
    
    try:
        driver = create_browser()
        if not driver:
            return photos
            
        logger.info(f"Loading page with browser: {url}")
        driver.get(url)
        
        WebDriverWait(driver, 10).until(
            EC.presence_of_element_located((By.TAG_NAME, "body"))
        )
        time.sleep(3)
        
        try:
            fotorama_nav = driver.find_element(By.CSS_SELECTOR, ".fotorama__nav-wrap.fotorama__nav-wrap--horizontal")
            nav_images = fotorama_nav.find_elements(By.TAG_NAME, "img")
            
            for img in nav_images:
                src = img.get_attribute("src") or img.get_attribute("data-src")
                if src:
                    photo_url = urljoin(BASE_URL, src)
                    if photo_url not in photos:
                        photos.append(photo_url)
                        
        except NoSuchElementException:
            logger.info("Fotorama gallery not found, trying alternative selectors")
        
        if not photos:
            gallery_selectors = [
                ".fotorama img",
                ".product-gallery img",
                ".gallery img",
                ".product-image-gallery img",
                "[class*='gallery'] img",
                "[class*='fotorama'] img"
            ]
            
            for selector in gallery_selectors:
                try:
                    images = driver.find_elements(By.CSS_SELECTOR, selector)
                    for img in images:
                        src = img.get_attribute("src") or img.get_attribute("data-src")
                        if src and "product" in src.lower():
                            photo_url = urljoin(BASE_URL, src)
                            if photo_url not in photos:
                                photos.append(photo_url)
                    if photos:
                        break
                except Exception:
                    continue
        
        if not photos:
            all_images = driver.find_elements(By.TAG_NAME, "img")
            for img in all_images:
                src = img.get_attribute("src") or img.get_attribute("data-src")
                if src and any(keyword in src.lower() for keyword in ["product", "item", "catalog"]):
                    photo_url = urljoin(BASE_URL, src)
                    if photo_url not in photos:
                        photos.append(photo_url)
        
        logger.info(f"Extracted {len(photos)} photos using browser")
        
    except Exception as e:
        logger.error(f"Error extracting photos with browser: {str(e)}")
    
    finally:
        if driver:
            driver.quit()
    
    return photos

def analyze_page_structure(url: str) -> Dict[str, Any]:
    """
    Analyze the actual HTML structure of a page to find correct selectors
    """
    driver = None
    try:
        driver = create_browser()
        if not driver:
            return {"error": "Failed to create browser"}
            
        logger.info(f"Analyzing page structure: {url}")
        driver.get(url)
        
        WebDriverWait(driver, 15).until(
            EC.presence_of_element_located((By.TAG_NAME, "body"))
        )
        time.sleep(5)
        
        html_content = driver.page_source
        soup = BeautifulSoup(html_content, 'html.parser')
        
        analysis = {
            "url": url,
            "title": soup.title.string if soup.title else "No title",
            "product_containers": [],
            "image_selectors": [],
            "name_selectors": [],
            "price_selectors": [],
            "description_selectors": []
        }
        
        potential_containers = soup.find_all(['div', 'li', 'article'], class_=True)
        
        for container in potential_containers:
            classes = ' '.join(container.get('class', []))
            
            if any(keyword in classes.lower() for keyword in ['product', 'item', 'result']):
                img = container.find('img')
                link = container.find('a', href=True)
                
                if img and link:
                    container_info = {
                        "selector": f".{classes.replace(' ', '.')}",
                        "classes": classes,
                        "has_image": bool(img),
                        "has_link": bool(link),
                        "image_src": img.get('src') if img else None,
                        "link_href": link.get('href') if link else None,
                        "text_content": container.get_text(strip=True)[:100] + "..." if len(container.get_text(strip=True)) > 100 else container.get_text(strip=True)
                    }
                    analysis["product_containers"].append(container_info)
        
        images = soup.find_all('img')
        for img in images[:10]:
            src = img.get('src') or img.get('data-src')
            if src and any(keyword in src.lower() for keyword in ['product', 'item', 'catalog']):
                parent_classes = []
                parent = img.parent
                while parent and len(parent_classes) < 3:
                    if parent.get('class'):
                        parent_classes.extend(parent.get('class'))
                    parent = parent.parent
                
                analysis["image_selectors"].append({
                    "img_src": src,
                    "img_alt": img.get('alt', ''),
                    "parent_classes": parent_classes,
                    "selector_path": f"{'.' + '.'.join(parent_classes[:2]) if parent_classes else ''} img"
                })
        
        return analysis
        
    except Exception as e:
        logger.error(f"Error analyzing page structure: {str(e)}")
        return {"error": str(e)}
    
    finally:
        if driver:
            driver.quit()