from fastapi import Query, HTTPException
from fastapi.responses import JSONResponse
from bs4 import BeautifulSoup
import asyncio
import aiohttp
from typing import List, Dict, Any

from http_utils import get_page_content_async, create_async_session
from translation import get_translation, translate_text
from price_utils import convert_price_to_eur
from database import get_products_db, apply_product_modifications

async def get_category_products_async(
    brand: str,
    page: int = 1,
    language: str = 'ru'
) -> Dict[str, Any]:
    """Get products from category page using async HTTP requests"""
    
    try:
        # Construct URL
        url = f"https://brandpage.ru/collection/{brand}?page={page}"
        
        # Get page content asynchronously
        session = await create_async_session()
        try:
            soup = await get_page_content_async(url, session)
            if not soup:
                raise HTTPException(status_code=500, detail="Failed to load category page")
            
            products = []
            
            # Find product containers
            product_selectors = [
                '.products-grid form.product-card',
                'form.product-card',
                '.product-card',
            ]
            
            product_containers = []
            for selector in product_selectors:
                containers = soup.select(selector)
                if containers:
                    product_containers = containers
                    break
            
            # Process products in parallel
            if product_containers:
                tasks = []
                for container in product_containers:
                    task = process_product_container_async(container, language, session)
                    tasks.append(task)
                
                # Execute all tasks concurrently
                results = await asyncio.gather(*tasks, return_exceptions=True)
                
                # Filter successful results
                for result in results:
                    if isinstance(result, dict) and result:
                        products.append(result)
            
        finally:
            await session.close()
        
        # Apply database modifications to each product
        products_db = get_products_db()
        modified_products = []
        for product in products:
            modified_product = apply_product_modifications(product, products_db)
            if modified_product:  # Only add if not deleted
                modified_products.append(modified_product)
        
        return {
            "products": modified_products,
            "page": page,
            "brand": brand,
            "total_found": len(modified_products),
            "language": language
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting category products: {str(e)}")

async def process_product_container_async(container, language: str, session: aiohttp.ClientSession) -> Dict[str, Any]:
    """Process individual product container asynchronously"""
    try:
        product = {}
        
        # Extract product link
        link_elem = container.find('a', href=lambda x: x and '/product/' in x)
        if link_elem:
            product_url = link_elem.get('href')
            if product_url and not product_url.startswith('http'):
                product_url = f"https://brandpage.ru{product_url}"
            product['url'] = product_url
            product['identifier'] = product_url.split('/')[-1] if product_url else None
        
        # Extract title
        title_selectors = [
            '.product-card__title',
            '.thumb__title',
            'h3',
            '.product-title',
            '[aria-label]'
        ]
        
        for selector in title_selectors:
            title_elem = container.select_one(selector)
            if title_elem:
                title = title_elem.get_text(strip=True) or title_elem.get('aria-label', '').strip()
                if title:
                    product['title'] = title
                    break
        
        # Extract price
        price_selectors = [
            '.price .price-item',
            '.product-card__price',
            '.price',
            '[class*="price"]'
        ]
        
        for selector in price_selectors:
            price_elem = container.select_one(selector)
            if price_elem:
                price_text = price_elem.get_text(strip=True)
                if price_text and any(char.isdigit() for char in price_text):
                    product['price'] = price_text
                    product['price_eur'] = convert_price_to_eur(price_text)
                    break
        
        # Extract image - get only one image URL
        img_selectors = [
            'img[data-srcset]',
            'img[src]',
            'img'
        ]
        
        for selector in img_selectors:
            img_elem = container.select_one(selector)
            if img_elem:
                img_url = img_elem.get('data-srcset') or img_elem.get('src')
                if img_url:
                    # Handle data-srcset which may contain multiple URLs
                    if 'data-srcset' in img_elem.attrs and img_url:
                        # Extract first URL from srcset (format: "url1 1x, url2 2x")
                        first_url = img_url.split(',')[0].strip().split(' ')[0]
                        img_url = first_url
                    
                    if not img_url.startswith('http'):
                        img_url = f"https://brandpage.ru{img_url}"
                    product['image'] = img_url
                    break
        
        # Translate title if needed
        if language != 'ru' and 'title' in product:
            try:
                translated_title = await translate_text_async(product['title'], language)
                if translated_title:
                    product['title_translated'] = translated_title
            except:
                pass  # Keep original title if translation fails
        
        return product if product else {}
        
    except Exception as e:
        return {}

async def translate_text_async(text: str, target_language: str) -> str:
    """Async wrapper for text translation"""
    # This would need to be implemented based on your translation service
    # For now, return sync version
    return translate_text(text, target_language)

# Sync wrapper for backward compatibility
def get_category_products(
    brand: str = Query(..., description="Brand name"), 
    page: int = Query(1, description="Page number"),
    language: str = Query('ru', description="Language code (ru, en, fr, de, he)")
):
    """Sync wrapper for async function"""
    return asyncio.run(get_category_products_async(brand, page, language))


async def search_products_async(
    query: str,
    page: int = 1,
    language: str = 'ru'
) -> Dict[str, Any]:
    """Search products by query using async HTTP requests"""
    
    try:
        # Construct URL based on page number
        if page == 1:
            url = f"https://brandpage.ru/collection/all?q={query}"
        else:
            url = f"https://brandpage.ru/collection/all?page={page}&q={query}"
        
        # Get page content asynchronously
        session = await create_async_session()
        try:
            soup = await get_page_content_async(url, session)
            if not soup:
                raise HTTPException(status_code=500, detail="Failed to load search page")
            
            products = []
            
            # Find product containers using same selectors as category products
            product_selectors = [
                '.products-grid form.product-card',
                'form.product-card',
                '.product-card',
            ]
            
            product_containers = []
            for selector in product_selectors:
                containers = soup.select(selector)
                if containers:
                    product_containers = containers
                    break
            
            # Process products in parallel
            if product_containers:
                tasks = []
                for container in product_containers:
                    task = process_product_container_async(container, language, session)
                    tasks.append(task)
                
                # Execute all tasks concurrently
                results = await asyncio.gather(*tasks, return_exceptions=True)
                
                # Filter successful results
                for result in results:
                    if isinstance(result, dict) and result:
                        products.append(result)
            
        finally:
            await session.close()
        
        # Apply database modifications to each product
        products_db = get_products_db()
        modified_products = []
        for product in products:
            modified_product = apply_product_modifications(product, products_db)
            if modified_product:  # Only add if not deleted
                modified_products.append(modified_product)
        
        return {
            "products": modified_products,
            "page": page,
            "query": query,
            "total_found": len(modified_products),
            "language": language
        }
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error searching products: {str(e)}")

# Sync wrapper for search function
def search_products(
    query: str = Query(..., description="Search query"), 
    page: int = Query(1, description="Page number"),
    language: str = Query('ru', description="Language code (ru, en, fr, de, he)")
):
    """Sync wrapper for async search function"""
    return asyncio.run(search_products_async(query, page, language))