#!/usr/bin/env python3
"""
Check what the site actually returns
"""

import requests
from bs4 import BeautifulSoup

def check_url_content(url, name):
    """Check what content a URL returns"""
    print(f"\n🔍 Checking {name}")
    print(f"URL: {url}")
    print("-" * 60)
    
    try:
        response = requests.get(url, timeout=10, headers={
            'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36'
        })
        
        print(f"Status Code: {response.status_code}")
        print(f"Content-Type: {response.headers.get('content-type', 'unknown')}")
        print(f"Content Length: {len(response.content)} bytes")
        
        if response.status_code == 200:
            soup = BeautifulSoup(response.content, 'html.parser')
            
            # Check title
            title = soup.title.string if soup.title else "No title"
            print(f"Page Title: {title}")
            
            # Look for common elements
            elements_to_check = [
                ('Products (.product-item)', '.product-item'),
                ('Products (.product)', '.product'),
                ('Items (.item)', '.item'),
                ('Links to .html', 'a[href$=".html"]'),
                ('Any links', 'a[href]'),
                ('Price elements', '[class*="price"]'),
                ('Add to cart', '[class*="cart"]'),
            ]
            
            print(f"\nElement counts:")
            for name, selector in elements_to_check:
                try:
                    count = len(soup.select(selector))
                    print(f"  {name}: {count}")
                except:
                    print(f"  {name}: Error")
            
            # Show some sample links
            links = soup.find_all('a', href=True)[:10]
            if links:
                print(f"\nSample links:")
                for i, link in enumerate(links[:5]):
                    href = link.get('href', '')[:50]
                    text = link.get_text(strip=True)[:30]
                    print(f"  {i+1}. {href} - {text}")
            
            # Check for error messages
            text_content = soup.get_text().lower()
            error_keywords = ['404', 'not found', 'error', 'no products', 'no results']
            found_errors = [kw for kw in error_keywords if kw in text_content]
            if found_errors:
                print(f"\n⚠️ Potential issues found: {found_errors}")
            
        else:
            print(f"❌ HTTP Error: {response.status_code}")
            print(f"Response text: {response.text[:200]}")
            
    except Exception as e:
        print(f"❌ Request failed: {e}")

def main():
    """Check different URLs"""
    print("🔍 Checking Site Response Content")
    print("=" * 80)
    
    urls_to_check = [
        ("https://febest.com.au/shop-parts.html?find=nissan", "Category - Nissan"),
        ("https://febest.com.au/catalogsearch/result/index/?q=nissan", "Search - Nissan"),
        ("https://febest.com.au/0133-001-febest-rear-leaf-spring-u-bolt-repairkit.html", "Product Page"),
        ("https://febest.com.au/", "Home Page"),
    ]
    
    for url, name in urls_to_check:
        check_url_content(url, name)

if __name__ == "__main__":
    main()