import json
import os
from config import PRODUCTS_DB_FILE

def load_products_db():
    """Load products database from JSON file"""
    if os.path.exists(PRODUCTS_DB_FILE):
        try:
            with open(PRODUCTS_DB_FILE, 'r', encoding='utf-8') as f:
                return json.load(f)
        except (json.JSONDecodeError, IOError):
            pass
    
    # Return default structure if file doesn't exist or is corrupted
    return {
        "created_products": {},
        "modified_products": {},
        "deleted_products": set()
    }

def save_products_db(db_data):
    """Save products database to JSON file"""
    # Convert set to list for JSON serialization
    db_copy = db_data.copy()
    db_copy["deleted_products"] = list(db_data["deleted_products"])
    
    with open(PRODUCTS_DB_FILE, 'w', encoding='utf-8') as f:
        json.dump(db_copy, f, ensure_ascii=False, indent=2)

def get_products_db():
    """Get products database with proper set conversion"""
    if os.path.exists(PRODUCTS_DB_FILE):
        try:
            with open(PRODUCTS_DB_FILE, 'r', encoding='utf-8') as f:
                db_data = json.load(f)
        except (json.JSONDecodeError, IOError):
            db_data = {
                "created_products": {},
                "modified_products": {},
                "deleted_products": set()
            }
    else:
        db_data = {
            "created_products": {},
            "modified_products": {},
            "deleted_products": set()
        }
    
    # Convert list back to set
    if isinstance(db_data.get("deleted_products"), list):
        db_data["deleted_products"] = set(db_data["deleted_products"])
    return db_data

def apply_product_modifications(product, products_db):
    """Apply modifications to a product from database"""
    product_id = product.get("product_identifier")
    if not product_id:
        return product
    
    # Check if product is deleted
    if product_id in products_db["deleted_products"]:
        return None
    
    # Apply modifications if they exist
    if product_id in products_db["modified_products"]:
        modifications = products_db["modified_products"][product_id]
        # Merge modifications with original product
        for key, value in modifications.items():
            if key != "updated_at":  # Don't overwrite system fields
                product[key] = value
    
    return product