#!/usr/bin/env python3
"""
Debug selectors on real febest.com.au pages
"""

import requests
from bs4 import BeautifulSoup
from selenium import webdriver
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
import time

def create_browser():
    """Create browser instance"""
    try:
        options = Options()
        options.add_argument('--headless')
        options.add_argument('--no-sandbox')
        options.add_argument('--disable-dev-shm-usage')
        options.add_argument('--disable-gpu')
        options.add_argument('--window-size=1920,1080')
        options.add_argument('--user-agent=Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36')
        
        driver = webdriver.Chrome(options=options)
        return driver
    except Exception as e:
        print(f"Failed to create browser: {e}")
        return None

def debug_page_selectors(url, page_type):
    """Debug selectors on a specific page"""
    print(f"\n🔍 Debugging {page_type}: {url}")
    print("=" * 80)
    
    browser = None
    try:
        browser = create_browser()
        if not browser:
            print("❌ Failed to create browser")
            return
        
        print("📄 Loading page...")
        browser.get(url)
        
        # Wait for page load
        WebDriverWait(browser, 15).until(
            lambda driver: driver.execute_script("return document.readyState") == "complete"
        )
        time.sleep(3)
        
        html_content = browser.page_source
        soup = BeautifulSoup(html_content, 'html.parser')
        
        print(f"✅ Page loaded - Content length: {len(html_content)}")
        print(f"📋 Page title: {soup.title.string if soup.title else 'No title'}")
        
        # Test different selectors
        selectors_to_test = [
            '.product-item-info',
            '.product-item',
            '.product-items .product-item',
            '.products .product-item',
            '.item',
            '.product',
            'a[href*=".html"]',
            '.product-name',
            '.product-item-name',
            '.price'
        ]
        
        print(f"\n🧪 Testing selectors:")
        print("-" * 40)
        
        found_elements = {}
        for selector in selectors_to_test:
            elements = soup.select(selector)
            count = len(elements)
            found_elements[selector] = count
            status = "✅" if count > 0 else "❌"
            print(f"{status} {selector}: {count} elements")
            
            # Show sample content for successful selectors
            if count > 0 and count <= 3:
                for i, elem in enumerate(elements[:2]):
                    text = elem.get_text(strip=True)[:100]
                    print(f"    Sample {i+1}: {text}...")
        
        # Look for any links ending with .html
        html_links = soup.find_all('a', href=lambda x: x and x.endswith('.html'))
        print(f"\n🔗 Found {len(html_links)} .html links")
        
        if html_links:
            for i, link in enumerate(html_links[:5]):
                href = link.get('href', '')
                text = link.get_text(strip=True)[:50]
                print(f"  {i+1}. {href} - {text}")
        
        # Check for common Magento classes
        magento_indicators = [
            'class*="product"', 'class*="item"', 'class*="catalog"',
            'data-product-id', 'class*="price"', 'class*="add-to-cart"'
        ]
        
        print(f"\n🛒 Magento indicators:")
        print("-" * 40)
        
        for indicator in magento_indicators:
            if '*=' in indicator:
                # Handle attribute contains selectors
                attr, value = indicator.split('*="')
                value = value.rstrip('"')
                elements = soup.find_all(attrs={attr.replace('class', 'class_'): lambda x: x and value in str(x)})
            else:
                elements = soup.select(f'[{indicator}]')
            
            count = len(elements)
            status = "✅" if count > 0 else "❌"
            print(f"{status} {indicator}: {count} elements")
        
        # Save page for manual inspection
        filename = f"debug_{page_type.replace(' ', '_').lower()}.html"
        with open(filename, 'w', encoding='utf-8') as f:
            f.write(html_content)
        print(f"\n💾 Page saved as: {filename}")
        
    except Exception as e:
        print(f"❌ Error debugging page: {e}")
    
    finally:
        if browser:
            browser.quit()

def main():
    """Debug different page types"""
    print("🚀 Debugging Selectors on febest.com.au")
    print("=" * 80)
    
    # Test different page types
    test_pages = [
        ("https://febest.com.au/shop-parts.html?find=nissan", "Category Page"),
        ("https://febest.com.au/catalogsearch/result/index/?q=nissan", "Search Page"),
        ("https://febest.com.au/0133-001-febest-rear-leaf-spring-u-bolt-repairkit.html", "Product Page")
    ]
    
    for url, page_type in test_pages:
        debug_page_selectors(url, page_type)
        time.sleep(2)  # Be respectful to the server
    
    print(f"\n🎯 SUMMARY")
    print("=" * 80)
    print("Check the saved HTML files to manually inspect the page structure:")
    print("- debug_category_page.html")
    print("- debug_search_page.html") 
    print("- debug_product_page.html")
    print("\nLook for actual CSS classes and structure used by the site.")

if __name__ == "__main__":
    main()