#!/usr/bin/env python3
"""
Final comprehensive test of all API endpoints
"""

import requests
import time
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

BASE_URL = "http://localhost:8000"

def test_health():
    """Test health endpoint"""
    print("=== Testing Health Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/health", timeout=10)
        print(f"Status: {response.status_code}")
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Health: {data.get('status')}")
            return True
        return False
    except Exception as e:
        print(f"❌ Health test failed: {e}")
        return False

def test_search():
    """Test search endpoint"""
    print("\n=== Testing Search Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/search", params={"query": "brake pad"}, timeout=30)
        print(f"Status: {response.status_code}")
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Search success: {data.get('success')}")
            print(f"   Total products: {data.get('total_products')}")
            if data.get('products'):
                print(f"   Sample: {data['products'][0].get('name', 'No name')}")
            return data.get('total_products', 0) > 0
        return False
    except Exception as e:
        print(f"❌ Search test failed: {e}")
        return False

def test_shop_by_mark():
    """Test shop-by-mark endpoint"""
    print("\n=== Testing Shop-by-Mark Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/shop-by-mark", params={"mark": "subaru"}, timeout=60)
        print(f"Status: {response.status_code}")
        if response.status_code == 200:
            data = response.json()
            print(f"✅ Shop-by-mark success: {data.get('success')}")
            print(f"   Total products: {data.get('total_products')}")
            print(f"   Method used: {data.get('method')}")
            print(f"   Successful URL: {data.get('successful_url')}")
            if data.get('products'):
                print(f"   Sample: {data['products'][0].get('name', 'No name')}")
            return data.get('total_products', 0) > 0
        return False
    except Exception as e:
        print(f"❌ Shop-by-mark test failed: {e}")
        return False

def test_product_details():
    """Test product details endpoint"""
    print("\n=== Testing Product Details Endpoint ===")
    
    # First get a product from search
    try:
        search_response = requests.get(f"{BASE_URL}/search", params={"query": "brake"}, timeout=30)
        if search_response.status_code == 200:
            search_data = search_response.json()
            if search_data.get('products'):
                # Use the first product's URL
                product_url = search_data['products'][0].get('url') or search_data['products'][0].get('slug')
                if product_url:
                    print(f"Testing with product URL: {product_url}")
                    
                    response = requests.get(f"{BASE_URL}/product-details", params={"url": product_url}, timeout=30)
                    print(f"Status: {response.status_code}")
                    if response.status_code == 200:
                        data = response.json()
                        print(f"✅ Product details success: {data.get('success')}")
                        if data.get('product'):
                            product = data['product']
                            print(f"   Name: {product.get('name', 'N/A')}")
                            print(f"   Price: {product.get('price', 'N/A')}")
                            print(f"   Images: {len(product.get('images', []))}")
                        return data.get('success', False)
                    return False
        
        print("⚠️ No products found to test details with")
        return False
        
    except Exception as e:
        print(f"❌ Product details test failed: {e}")
        return False

def main():
    """Run all tests"""
    print("🚀 Running Final Comprehensive Test...")
    
    # Wait a moment for server to be ready
    time.sleep(2)
    
    results = {
        'health': test_health(),
        'search': test_search(),
        'shop_by_mark': test_shop_by_mark(),
        'product_details': test_product_details()
    }
    
    print(f"\n=== FINAL RESULTS ===")
    for test_name, result in results.items():
        status = "✅ PASS" if result else "❌ FAIL"
        print(f"{test_name.replace('_', ' ').title()}: {status}")
    
    total_passed = sum(results.values())
    print(f"\nOverall: {total_passed}/4 tests passed")
    
    if total_passed == 4:
        print("🎉 ALL TESTS PASSED! API is working correctly!")
    elif total_passed >= 2:
        print("⚠️ Most tests passed, minor issues remain")
    else:
        print("❌ Major issues detected, needs more work")

if __name__ == "__main__":
    main()