#!/usr/bin/env python3
"""
Find correct category structure on febest.com.au
"""

from browser_utils import create_browser
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from bs4 import BeautifulSoup
import logging
import re

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def explore_main_page():
    """Explore main page to find category structure"""
    print("=== Exploring Main Page ===")
    
    browser = None
    try:
        browser = create_browser()
        if not browser:
            print("❌ Failed to create browser")
            return []
            
        browser.get("https://febest.com.au")
        
        WebDriverWait(browser, 15).until(
            EC.presence_of_element_located((By.TAG_NAME, "body"))
        )
        
        soup = BeautifulSoup(browser.page_source, 'html.parser')
        
        # Look for navigation menus
        nav_selectors = [
            'nav', '.navigation', '.menu', '.navbar',
            '.main-menu', '.category-menu', '.nav-menu'
        ]
        
        categories = []
        
        for selector in nav_selectors:
            nav_elements = soup.select(selector)
            for nav in nav_elements:
                # Find links that might be categories
                links = nav.find_all('a', href=True)
                for link in links:
                    href = link.get('href')
                    text = link.get_text(strip=True).lower()
                    
                    # Look for car brand names
                    car_brands = [
                        'subaru', 'toyota', 'honda', 'nissan', 'mazda',
                        'mitsubishi', 'suzuki', 'ford', 'holden', 'bmw',
                        'mercedes', 'audi', 'volkswagen', 'hyundai', 'kia'
                    ]
                    
                    if any(brand in text for brand in car_brands):
                        categories.append({
                            'text': text,
                            'href': href,
                            'full_url': href if href.startswith('http') else f"https://febest.com.au{href}"
                        })
        
        # Also look for any links containing car brands
        all_links = soup.find_all('a', href=True)
        for link in all_links:
            href = link.get('href')
            text = link.get_text(strip=True).lower()
            
            if 'subaru' in text or 'subaru' in href:
                categories.append({
                    'text': text,
                    'href': href,
                    'full_url': href if href.startswith('http') else f"https://febest.com.au{href}"
                })
        
        # Save page for manual inspection
        with open('main_page_debug.html', 'w', encoding='utf-8') as f:
            f.write(browser.page_source)
        
        return categories
        
    except Exception as e:
        print(f"❌ Error: {e}")
        return []
    finally:
        if browser:
            browser.quit()

def test_search_for_categories():
    """Test search functionality to find categories"""
    print("\n=== Testing Search for Categories ===")
    
    browser = None
    try:
        browser = create_browser()
        if not browser:
            return []
            
        # Try searching for "subaru" to see if it shows category results
        search_url = "https://febest.com.au/catalogsearch/result/?q=subaru"
        browser.get(search_url)
        
        WebDriverWait(browser, 15).until(
            EC.presence_of_element_located((By.TAG_NAME, "body"))
        )
        
        soup = BeautifulSoup(browser.page_source, 'html.parser')
        
        # Look for category suggestions or filters
        category_links = []
        
        # Check for category filters or suggestions
        filter_selectors = [
            '.filter-options', '.layered-filter', '.category-filter',
            '.search-suggestions', '.category-suggestions'
        ]
        
        for selector in filter_selectors:
            elements = soup.select(selector)
            for elem in elements:
                links = elem.find_all('a', href=True)
                for link in links:
                    href = link.get('href')
                    text = link.get_text(strip=True)
                    if 'subaru' in text.lower() or 'subaru' in href.lower():
                        category_links.append({
                            'text': text,
                            'href': href,
                            'full_url': href if href.startswith('http') else f"https://febest.com.au{href}"
                        })
        
        # Save search page for inspection
        with open('search_subaru_debug.html', 'w', encoding='utf-8') as f:
            f.write(browser.page_source)
        
        return category_links
        
    except Exception as e:
        print(f"❌ Error: {e}")
        return []
    finally:
        if browser:
            browser.quit()

def main():
    """Main exploration function"""
    print("🔍 Exploring febest.com.au to find correct category structure...")
    
    # Explore main page
    main_categories = explore_main_page()
    
    # Test search
    search_categories = test_search_for_categories()
    
    print(f"\n=== RESULTS ===")
    print(f"Found {len(main_categories)} potential categories from main page")
    print(f"Found {len(search_categories)} potential categories from search")
    
    all_categories = main_categories + search_categories
    
    if all_categories:
        print("\n🎯 Potential Subaru category URLs:")
        seen_urls = set()
        for cat in all_categories:
            if cat['full_url'] not in seen_urls:
                print(f"  • {cat['text']} -> {cat['full_url']}")
                seen_urls.add(cat['full_url'])
    else:
        print("\n❌ No category URLs found!")
        print("💡 Suggestions:")
        print("1. Check main_page_debug.html for manual inspection")
        print("2. Check search_subaru_debug.html for search results")
        print("3. The site might use AJAX/JavaScript for categories")
        print("4. Categories might be in a different format")

if __name__ == "__main__":
    main()