from fastapi import Query, HTTPException
from bs4 import BeautifulSoup
import time
import asyncio
import aiohttp

from http_utils import get_page_content, get_page_content_async, create_async_session
from translation import translate_text

def get_header_categories(language: str = Query('ru', description="Language code (ru, en, fr, de, he)")):
    """Sync wrapper for async function with new format"""
    return asyncio.run(get_header_categories_async(language))
    
    """Get all categories from header navigation on brandpage.ru using HTTP requests"""
    
    try:
        # Get page content
        soup = get_page_content('https://brandpage.ru')
        if not soup:
            raise HTTPException(status_code=500, detail="Failed to load page")
        
        header_categories = []
        
        # Более специфичные селекторы для поиска навигации
        nav_selectors = [
            'nav[class*="navigation"]',
            'nav[class*="menu"]',
            '.header-navigation',
            '.main-navigation',
            '.site-navigation',
            'nav',
            'header nav',
            '[class*="nav-menu"]'
        ]
        
        header_nav = None
        for selector in nav_selectors:
            header_nav = soup.select_one(selector)
            if header_nav:
                break
        
        if header_nav:
            # Более точные селекторы для элементов меню
            menu_item_selectors = [
                'li[class*="menu-item"]',
                'li[class*="nav-item"]', 
                '.menu-item',
                '.nav-item',
                'li:has(a)',
                'li'
            ]
            
            main_nav_items = []
            for selector in menu_item_selectors:
                main_nav_items = header_nav.select(selector)
                if main_nav_items:
                    break
            
            for nav_item in main_nav_items:
                # Get main category link
                main_link = nav_item.find('a')
                if main_link and main_link.get_text(strip=True):
                    category_name = main_link.get_text(strip=True)
                    category_url = main_link.get('href', '')
                    
                    # Skip non-category items
                    skip_keywords = ['contact', 'about', 'login', 'register', 'search', 'cart', 'account', 'корзина', 'контакты', 'о нас']
                    if any(keyword in category_name.lower() for keyword in skip_keywords):
                        continue
                    
                    category_data = {
                        'name': category_name,
                        'url': category_url,
                        'subcategories': []
                    }
                    
                    # Улучшенный поиск подкатегорий
                    subcategory_selectors = [
                        'ul.dropdown-menu',
                        'ul.sub-menu', 
                        'ul.submenu',
                        '.dropdown-content',
                        '.mega-menu',
                        'div[class*="dropdown"] ul',
                        'div[class*="submenu"] ul',
                        'ul:not(.menu):not(.navigation)',
                        'ul'
                    ]
                    
                    # Ищем подкатегории внутри текущего элемента меню
                    for sub_selector in subcategory_selectors:
                        submenu = nav_item.select_one(sub_selector)
                        if submenu:
                            # Находим все ссылки в подменю
                            sub_links = submenu.find_all('a')
                            for sub_link in sub_links:
                                sub_text = sub_link.get_text(strip=True)
                                sub_url = sub_link.get('href', '')
                                
                                # Проверяем что это не дубликат основной категории
                                if sub_text and sub_text != category_name and len(sub_text) > 1:
                                    category_data['subcategories'].append({
                                        'name': sub_text,
                                        'url': sub_url
                                    })
                            
                            # Если нашли подкатегории, прекращаем поиск
                            if category_data['subcategories']:
                                break
                    
                    # Альтернативный поиск: ищем следующие элементы li на том же уровне
                    if not category_data['subcategories']:
                        # Проверяем есть ли вложенные li элементы
                        nested_items = nav_item.find_all('li')
                        for nested_item in nested_items:
                            if nested_item != nav_item:  # Исключаем сам элемент
                                nested_link = nested_item.find('a')
                                if nested_link:
                                    sub_text = nested_link.get_text(strip=True)
                                    sub_url = nested_link.get('href', '')
                                    if sub_text and sub_text != category_name:
                                        category_data['subcategories'].append({
                                            'name': sub_text,
                                            'url': sub_url
                                        })
                    
                    # Translate if needed
                    if language != 'ru':
                        category_data['name'] = translate_text(category_data['name'], language)
                        for sub in category_data['subcategories']:
                            sub['name'] = translate_text(sub['name'], language)
                    
                    header_categories.append(category_data)
        
        return {"header_categories": header_categories}
    
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting header categories: {str(e)}")

async def get_header_categories_async(language: str = 'ru'):
    """Get all categories from header navigation with subcategories in new format"""
    
    try:
        # Get page content asynchronously
        session = await create_async_session()
        try:
            soup = await get_page_content_async('https://brandpage.ru', session)
            if not soup:
                raise HTTPException(status_code=500, detail="Failed to load page")
            
            header_categories = []
            
            # Find navigation selectors
            nav_selectors = [
                'nav[class*="navigation"]',
                'nav[class*="menu"]',
                '.header-navigation',
                '.main-navigation',
                '.site-navigation',
                'nav',
                'header nav',
                '[class*="nav-menu"]'
            ]
            
            header_nav = None
            for selector in nav_selectors:
                header_nav = soup.select_one(selector)
                if header_nav:
                    break
            
            if header_nav:
                # Find menu items
                menu_item_selectors = [
                    'li[class*="menu-item"]',
                    'li[class*="nav-item"]', 
                    '.menu-item',
                    '.nav-item',
                    'li:has(a)',
                    'li'
                ]
                
                main_nav_items = []
                for selector in menu_item_selectors:
                    main_nav_items = header_nav.select(selector)
                    if main_nav_items:
                        break
                
                for nav_item in main_nav_items:
                    # Get main category link
                    main_link = nav_item.find('a')
                    if main_link and main_link.get_text(strip=True):
                        category_name = main_link.get_text(strip=True)
                        category_url = main_link.get('href', '')
                        
                        # Skip non-category items
                        skip_keywords = ['contact', 'about', 'login', 'register', 'search', 'cart', 'account', 'корзина', 'контакты', 'о нас']
                        if any(keyword in category_name.lower() for keyword in skip_keywords):
                            continue
                        
                        # Extract URL slug from full URL
                        url_slug = category_url
                        if category_url.startswith('http'):
                            url_slug = category_url.split('/')[-1] or category_url.split('/')[-2]
                        elif category_url.startswith('/'):
                            url_slug = category_url.lstrip('/')
                        
                        category_data = {
                            'name': category_name,
                            'url': url_slug,
                            'subcategories': []
                        }
                        
                        # Find subcategories
                        subcategory_selectors = [
                            'ul.dropdown-menu',
                            'ul.sub-menu', 
                            'ul.submenu',
                            '.dropdown-content',
                            '.mega-menu',
                            'div[class*="dropdown"] ul',
                            'div[class*="submenu"] ul',
                            'ul:not(.menu):not(.navigation)',
                            'ul'
                        ]
                        
                        # Search for subcategories within current menu item
                        for sub_selector in subcategory_selectors:
                            submenu = nav_item.select_one(sub_selector)
                            if submenu:
                                sub_links = submenu.find_all('a')
                                for sub_link in sub_links:
                                    sub_text = sub_link.get_text(strip=True)
                                    sub_url = sub_link.get('href', '')
                                    
                                    # Extract URL slug for subcategory
                                    sub_url_slug = sub_url
                                    if sub_url.startswith('http'):
                                        sub_url_slug = sub_url.split('/')[-1] or sub_url.split('/')[-2]
                                    elif sub_url.startswith('/'):
                                        sub_url_slug = sub_url.lstrip('/')
                                    
                                    # Check that it's not duplicate of main category
                                    if sub_text and sub_text != category_name and len(sub_text) > 1:
                                        category_data['subcategories'].append({
                                            'name': sub_text,
                                            'url': sub_url_slug
                                        })
                                
                                # If found subcategories, stop searching
                                if category_data['subcategories']:
                                    break
                        
                        # Alternative search: look for nested li elements
                        if not category_data['subcategories']:
                            nested_items = nav_item.find_all('li')
                            for nested_item in nested_items:
                                if nested_item != nav_item:  # Exclude the item itself
                                    nested_link = nested_item.find('a')
                                    if nested_link:
                                        sub_text = nested_link.get_text(strip=True)
                                        sub_url = nested_link.get('href', '')
                                        
                                        # Extract URL slug
                                        sub_url_slug = sub_url
                                        if sub_url.startswith('http'):
                                            sub_url_slug = sub_url.split('/')[-1] or sub_url.split('/')[-2]
                                        elif sub_url.startswith('/'):
                                            sub_url_slug = sub_url.lstrip('/')
                                        
                                        if sub_text and sub_text != category_name:
                                            category_data['subcategories'].append({
                                                'name': sub_text,
                                                'url': sub_url_slug
                                            })
                        
                        # Translate if needed
                        if language != 'ru':
                            category_data['name'] = translate_text(category_data['name'], language)
                            for sub in category_data['subcategories']:
                                sub['name'] = translate_text(sub['name'], language)
                        
                        header_categories.append(category_data)
            
            return header_categories
            
        finally:
            await session.close()
            
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting header categories: {str(e)}")