import aiohttp
import asyncio
from bs4 import BeautifulSoup
from typing import Optional

async def create_async_session():
    """Create async HTTP session with optimized settings"""
    connector = aiohttp.TCPConnector(
        limit=100,
        limit_per_host=30,
        ttl_dns_cache=300,
        use_dns_cache=True,
    )
    
    timeout = aiohttp.ClientTimeout(total=10, connect=5)
    
    headers = {
        'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
        'Accept-Language': 'ru-RU,ru;q=0.9,en;q=0.8',
        'Accept-Encoding': 'gzip, deflate, br',
        'Connection': 'keep-alive',
        'Upgrade-Insecure-Requests': '1',
    }
    
    return aiohttp.ClientSession(
        connector=connector,
        timeout=timeout,
        headers=headers
    )

async def get_page_content_async(url: str, session: aiohttp.ClientSession = None, retries: int = 3, delay: float = 1.0) -> Optional[BeautifulSoup]:
    """Get page content using aiohttp and parse with BeautifulSoup"""
    close_session = False
    if session is None:
        session = await create_async_session()
        close_session = True
    
    try:
        for attempt in range(retries):
            try:
                async with session.get(url) as response:
                    response.raise_for_status()
                    content = await response.read()
                    
                    # Parse with BeautifulSoup
                    soup = BeautifulSoup(content, 'html.parser')
                    return soup
                    
            except aiohttp.ClientError as e:
                if attempt < retries - 1:
                    await asyncio.sleep(delay * (attempt + 1))
                    continue
                else:
                    raise Exception(f"Failed to fetch {url} after {retries} attempts: {str(e)}")
    finally:
        if close_session:
            await session.close()
    
    return None

# Keep sync version for backward compatibility
def get_page_content(url: str, retries: int = 3, delay: float = 1.0) -> Optional[BeautifulSoup]:
    """Sync wrapper for async function"""
    return asyncio.run(get_page_content_async(url, retries=retries, delay=delay))

def wait_for_element(soup: BeautifulSoup, selector: str, timeout: int = 5) -> bool:
    """Simple wait simulation - just check if element exists"""
    element = soup.select_one(selector)
    return element is not None

# Backward compatibility aliases
cleanup_firefox_profile = lambda x: None
get_firefox_driver_for_environment = lambda: {'binary_location': None, 'driver_path': None}