import requests
import re

def get_exchange_rate_to_eur():
    """
    Get current RUB to EUR exchange rate from exchangerate-api.com
    
    Returns:
        float: Exchange rate from RUB to EUR
    """
    try:
        response = requests.get("https://api.exchangerate-api.com/v4/latest/RUB")
        data = response.json()
        return data['rates']['EUR']
    except:
        # Fallback rate if API is unavailable
        return 0.011  # Approximate rate

def convert_price_to_eur(price_text):
    """
    Convert price from RUB to EUR
    
    Args:
        price_text: Price string in RUB (e.g., "32 500 ₽")
    
    Returns:
        tuple: (EUR price as float, original currency)
    """
    if not price_text:
        return None, None
    
    # Remove currency symbols and clean the text
    clean_text = price_text.replace('₽', '').replace('руб', '').replace('рубл', '').replace('€', '').replace('EUR', '').replace('Price', '')
    
    # Extract numeric value from price string
    # Handle both comma and dot as decimal separators
    price_numbers = re.findall(r'[\d\s,\.]+', clean_text)
    if not price_numbers:
        return None, price_text
    
    try:
        # Get numeric price value
        price_str = price_numbers[0].replace(' ', '').replace(',', '.')
        price_value = float(price_str)
        
        # Check if price is already in EUR (if original text contained € or EUR)
        if '€' in price_text or 'EUR' in price_text.upper():
            return price_value, "EUR"
        
        # Get exchange rate and convert from RUB to EUR
        exchange_rate = get_exchange_rate_to_eur()
        eur_price = round(price_value * exchange_rate, 2)
        
        return eur_price, "RUB"
    except:
        return None, price_text

def convert_prices_in_text(text):
    """
    Convert all prices in text from RUB to EUR
    
    Args:
        text: Text containing prices in RUB
    
    Returns:
        str: Text with prices converted to EUR
    """
    if not text:
        return text
    
    # Pattern to find prices in text (e.g., "1000 ₽", "1 000 руб", "1000 рублей", "Price228.9 €")
    price_patterns = [
        r'Price([\d\s,\.]+)\s*€',  # Handle "Price228.9 €" format
        r'([\d\s,\.]+)\s*(?:₽|руб|рубл[ьей]*)',  # Handle RUB prices
        r'([\d\s,\.]+)\s*€'  # Handle EUR prices without "Price" prefix
    ]
    
    result_text = text
    
    for pattern in price_patterns:
        def replace_price(match):
            price_str = match.group(1)
            
            # If it's already EUR (from patterns with €), just format it properly
            if '€' in match.group(0) and 'Price' in match.group(0):
                # Remove "Price" prefix and add space before €
                try:
                    price_value = float(price_str.replace(' ', '').replace(',', '.'))
                    return f" {price_value} €"
                except:
                    return f" {price_str} €"
            elif '€' in match.group(0):
                # Just add space if needed
                try:
                    price_value = float(price_str.replace(' ', '').replace(',', '.'))
                    return f" {price_value} €"
                except:
                    return f" {price_str} €"
            else:
                # Convert from RUB to EUR
                eur_price, _ = convert_price_to_eur(price_str + " ₽")
                if eur_price is not None:
                    return f" {eur_price} €"
                return match.group(0)
        
        result_text = re.sub(pattern, replace_price, result_text, flags=re.IGNORECASE)
    
    return result_text