from fastapi import HTTPException
from fastapi.responses import JSONResponse
from bs4 import BeautifulSoup
import time
import re
import asyncio
import aiohttp

from http_utils import get_page_content, get_page_content_async, create_async_session
from translation import translate_text
from price_utils import convert_price_to_eur, convert_prices_in_text
from database import get_products_db

async def get_product_details_async(product_identifier: str, language: str = 'ru'):
    """Get product details from brandpage.ru using async HTTP requests"""
    
    try:
        # Construct URL
        url = f"https://brandpage.ru/product/{product_identifier}"
        
        # Get page content asynchronously
        session = await create_async_session()
        try:
            soup = await get_page_content_async(url, session)
            if not soup:
                raise HTTPException(status_code=500, detail="Failed to load product page")
            
            # Extract product information (same logic as sync version)
            product_data = {}
            
            # Title - добавлены более универсальные селекторы
            title_selectors = [
                '.product__title',
                '.product-meta__title', 
                'h1.product-title',
                'h1[class*="title"]',
                'h1[class*="product"]',
                '.product-title',
                '.title',
                'h1',
                '.page-title',
                '[class*="product-name"]',
                '[class*="item-title"]'
            ]
            
            for selector in title_selectors:
                title_elem = soup.select_one(selector)
                if title_elem:
                    product_data['title'] = title_elem.get_text(strip=True)
                    break
            
            # Price - берем цену из блока product__price
            price_selectors = [
                '.product__price',
                '.product__price .price-item',
                '.product__price span',
                '.product__price div'
            ]
            
            price_rub = None
            for selector in price_selectors:
                price_elem = soup.select_one(selector)
                if price_elem:
                    price_text = price_elem.get_text(strip=True)
                    if price_text and any(char.isdigit() for char in price_text):
                        price_rub = price_text
                        break
            
            # Description - улучшенные селекторы
            description_selectors = [
                '.product__description',
                '.product-description',
                '.product-meta__description',
                '.description',
                '[class*="description"]',
                '.product-details',
                '.product-info',
                '.product-content',
                '.item-description',
                'div[class*="desc"]',
                '.product-summary'
            ]
            
            description = None
            for selector in description_selectors:
                desc_elem = soup.select_one(selector)
                if desc_elem:
                    description = desc_elem.get_text(strip=True)
                    break
            
            # Images - собираем массив изображений
            image_selectors = [
                '.product__media img',
                '.product-images img',
                '.product-gallery img',
                '.product-image img',
                'img[src*="product"]',
                '.main-image img',
                '.featured-image img',
                'img[class*="product"]',
                'img[alt*="product"]',
                '.product-photo img',
                'img'
            ]
            
            images = []
            for selector in image_selectors:
                img_elements = soup.select(selector)
                for img in img_elements:
                    src = img.get('src') or img.get('data-src') or img.get('data-original')
                    if src:
                        if not src.startswith('http'):
                            src = f"https://brandpage.ru{src}"
                        if src not in images:  # Избегаем дубликатов
                            images.append(src)
                if images:
                    break
            
            # Availability - расширенные селекторы
            availability_selectors = [
                '.product__availability',
                '.availability',
                '.stock-status',
                '[class*="stock"]',
                '[class*="availability"]',
                '.in-stock',
                '.out-of-stock',
                '.product-availability',
                '.inventory-status',
                '[data-availability]'
            ]
            
            availability = None
            for selector in availability_selectors:
                avail_elem = soup.select_one(selector)
                if avail_elem:
                    availability = avail_elem.get_text(strip=True)
                    break
            
            # Sizes - улучшенные селекторы
            size_selectors = [
                '.product__variants select option',
                '.size-selector option',
                '.variant-selector option',
                'select[name*="size"] option',
                'select[class*="size"] option',
                '.size-options option',
                '.product-options select option',
                '[data-variant-option] option'
            ]
            
            sizes = []
            for selector in size_selectors:
                size_elements = soup.select(selector)
                for size_elem in size_elements:
                    size_text = size_elem.get_text(strip=True)
                    if size_text and size_text.lower() not in ['выберите размер', 'select size', '', 'размер']:
                        sizes.append(size_text)
                if sizes:
                    break
            
            # Final product data - возвращаем images как массив
            final_product_data = {
                'title': product_data.get('title', 'Title not available'),
                'price': product_data.get('price', 'Price not available'),
                'price_eur': None,
                'description': description or 'Description not available',
                'images': images,  # Массив изображений
                'availability': availability or 'Availability not specified',
                'sizes': sizes,
                'url': url
            }
            
            # Convert price to EUR immediately
            price_eur = None
            if price_rub:
                try:
                    price_eur = convert_price_to_eur(price_rub)
                except:
                    pass
            
            # Final product data - убираем price в рублях, оставляем только price_eur
            final_product_data = {
                'title': product_data.get('title', 'Title not available'),
                'price_eur': price_eur,  # Только цена в евро
                'description': description or 'Description not available',
                'images': images,
                'availability': availability or 'Availability not specified',
                'sizes': sizes,
                'url': url
            }
            
            # Убираем блок конвертации цены, так как уже сделали выше
            # if final_product_data['price'] and final_product_data['price'] != 'Price not available':
            #     try:
            #         final_product_data['price_eur'] = convert_price_to_eur(final_product_data['price'])
            #     except:
            #         pass
            
            # Translate text fields if language is not Russian
            if language != 'ru':
                if final_product_data['title'] and final_product_data['title'] != 'Title not available':
                    final_product_data['title'] = translate_text(final_product_data['title'], language)
                if description:
                    final_product_data['description'] = translate_text(description, language)
                if availability:
                    final_product_data['availability'] = translate_text(availability, language)
                
                # Translate sizes
                if sizes:
                    final_product_data['sizes'] = [translate_text(size, language) for size in sizes]
            
            # Convert prices in description text
            if final_product_data['description']:
                final_product_data['description'] = convert_prices_in_text(final_product_data['description'])
            
            return JSONResponse(content={
                "product": final_product_data,
                "language": language,
                "source": "brandpage.ru"
            })
            
        finally:
            await session.close()
            
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error getting product details: {str(e)}")