import os
import tempfile
from selenium import webdriver
from selenium.webdriver.firefox.options import Options
from selenium.webdriver.firefox.service import Service
from config import FIREFOX_BINARY_PATH, GECKODRIVER_PATH

def create_firefox_driver_with_unique_profile():
    """Create Firefox WebDriver with minimal configuration for maximum stability"""
    firefox_options = Options()
    
    # ТОЛЬКО базовые опции
    firefox_options.add_argument('--headless')
    firefox_options.add_argument('--no-sandbox')
    firefox_options.add_argument('--disable-dev-shm-usage')
    firefox_options.add_argument('--disable-gpu')
    
    # КРИТИЧЕСКИ ВАЖНЫЕ настройки для стабильности
    firefox_options.set_preference('marionette.enabled', True)
    firefox_options.set_preference('marionette.port', 2828)
    firefox_options.set_preference('devtools.debugger.remote-enabled', False)
    firefox_options.set_preference('devtools.chrome.enabled', False)
    
    # Отключаем ВСЕ ненужное
    firefox_options.set_preference('browser.safebrowsing.enabled', False)
    firefox_options.set_preference('browser.safebrowsing.malware.enabled', False)
    firefox_options.set_preference('media.volume_scale', '0.0')
    firefox_options.set_preference('dom.webnotifications.enabled', False)
    firefox_options.set_preference('dom.push.enabled', False)
    
    # АГРЕССИВНЫЕ таймауты
    firefox_options.set_preference('network.http.connection-timeout', 5)
    firefox_options.set_preference('network.http.response.timeout', 5)
    firefox_options.set_preference('dom.max_script_run_time', 3)
    firefox_options.set_preference('dom.max_chrome_script_run_time', 3)
    
    # Отключаем кэш полностью
    firefox_options.set_preference('browser.cache.disk.enable', False)
    firefox_options.set_preference('browser.cache.memory.enable', False)
    firefox_options.set_preference('network.http.use-cache', False)
    
    # Создаем временную директорию
    profiles_dir = '/tmp/firefox-profiles'
    os.makedirs(profiles_dir, exist_ok=True)
    temp_dir = tempfile.mkdtemp(prefix='firefox_', dir=profiles_dir)
    firefox_options.add_argument(f'--profile={temp_dir}')
    
    # Устанавливаем бинарник Firefox
    if FIREFOX_BINARY_PATH and os.path.exists(FIREFOX_BINARY_PATH):
        firefox_options.binary_location = FIREFOX_BINARY_PATH
    
    # ИСПРАВЛЕННАЯ конфигурация service для Docker
    service_args = [
        '--log', 'fatal',
        '--host', '0.0.0.0',  # КЛЮЧЕВОЕ ИЗМЕНЕНИЕ: используем 0.0.0.0 вместо localhost
        '--port', '4444',
        '--marionette-port', '2828'
    ]
    
    if GECKODRIVER_PATH and os.path.exists(GECKODRIVER_PATH):
        service = Service(GECKODRIVER_PATH, service_args=service_args)
    else:
        service = Service(service_args=service_args)
    
    try:
        driver = webdriver.Firefox(service=service, options=firefox_options)
        
        # ЭКСТРЕМАЛЬНО короткие таймауты
        driver.set_page_load_timeout(5)
        driver.implicitly_wait(2)
        driver.set_script_timeout(3)
        
        return driver, temp_dir
    except Exception as e:
        # Очистка при ошибке
        import shutil
        if os.path.exists(temp_dir):
            shutil.rmtree(temp_dir, ignore_errors=True)
        raise Exception(f"Failed to create Firefox driver: {str(e)}")

def cleanup_firefox_profile(temp_dir):
    """Clean up temporary Firefox profile directory"""
    import shutil
    if temp_dir and os.path.exists(temp_dir):
        try:
            shutil.rmtree(temp_dir, ignore_errors=True)
        except Exception:
            pass

def get_firefox_driver_for_environment():
    """Get Firefox driver configuration for current environment"""
    return {
        'binary_location': FIREFOX_BINARY_PATH,
        'driver_path': GECKODRIVER_PATH
    }

def get_optimized_firefox_options():
    """Get optimized Firefox options for fast execution"""
    firefox_options = Options()
    
    # Basic options
    firefox_options.add_argument('--headless')
    firefox_options.add_argument('--no-sandbox')
    firefox_options.add_argument('--disable-dev-shm-usage')
    firefox_options.add_argument('--disable-gpu')
    firefox_options.add_argument('--disable-extensions')
    firefox_options.add_argument('--disable-plugins')
    firefox_options.add_argument('--disable-images')
    
    # Performance optimizations
    firefox_options.set_preference('media.volume_scale', '0.0')
    firefox_options.set_preference('dom.webnotifications.enabled', False)
    firefox_options.set_preference('dom.push.enabled', False)
    firefox_options.set_preference('browser.safebrowsing.enabled', False)
    firefox_options.set_preference('browser.safebrowsing.malware.enabled', False)
    firefox_options.set_preference('browser.safebrowsing.phishing.enabled', False)
    firefox_options.set_preference('network.http.pipelining', True)
    firefox_options.set_preference('network.http.proxy.pipelining', True)
    firefox_options.set_preference('network.http.pipelining.maxrequests', 8)
    firefox_options.set_preference('content.notify.interval', 500000)
    firefox_options.set_preference('content.notify.ontimer', True)
    firefox_options.set_preference('content.switch.threshold', 250000)
    firefox_options.set_preference('browser.cache.memory.capacity', 65536)
    firefox_options.set_preference('browser.startup.homepage', 'about:blank')
    firefox_options.set_preference('browser.startup.page', 0)
    firefox_options.set_preference('browser.cache.disk.enable', False)
    firefox_options.set_preference('browser.cache.memory.enable', False)
    firefox_options.set_preference('browser.cache.offline.enable', False)
    firefox_options.set_preference('network.http.use-cache', False)
    
    return firefox_options

def setup_fast_timeouts(driver):
    """Setup aggressive timeouts for Firefox driver"""
    driver.set_page_load_timeout(10)
    driver.implicitly_wait(2)
    driver.set_script_timeout(5)

# Aliases for backward compatibility with Chrome function names
get_chrome_driver_for_environment = get_firefox_driver_for_environment
get_optimized_chrome_options = get_optimized_firefox_options
create_chrome_driver_with_unique_profile = create_firefox_driver_with_unique_profile
cleanup_chrome_profile = cleanup_firefox_profile