#!/usr/bin/env python3
"""
Test script for API endpoints
"""
import requests
import json
from datetime import datetime

BASE_URL = "http://localhost:8000"

def test_health_endpoint():
    """Test health check endpoint"""
    try:
        response = requests.get(f"{BASE_URL}/health", timeout=10)
        if response.status_code == 200:
            print("✅ Health check: OK")
            print(f"   Response: {response.json()}")
            return True
        else:
            print(f"❌ Health check failed: {response.status_code}")
            return False
    except Exception as e:
        print(f"❌ Health check error: {str(e)}")
        return False

def test_search_endpoint():
    """Test search endpoint"""
    try:
        params = {"query": "brake"}
        response = requests.get(f"{BASE_URL}/search", params=params, timeout=30)
        if response.status_code == 200:
            data = response.json()
            print("✅ Search endpoint: OK")
            print(f"   Found {data.get('total_products', 0)} products")
            return True
        else:
            print(f"❌ Search endpoint failed: {response.status_code}")
            print(f"   Response: {response.text}")
            return False
    except Exception as e:
        print(f"❌ Search endpoint error: {str(e)}")
        return False

def test_shop_by_mark_endpoint():
    """Test shop by mark endpoint"""
    try:
        params = {"mark": "toyota"}
        response = requests.get(f"{BASE_URL}/shop-by-mark", params=params, timeout=30)
        if response.status_code == 200:
            data = response.json()
            print("✅ Shop by mark endpoint: OK")
            print(f"   Found {data.get('total_products', 0)} products for Toyota")
            return True
        else:
            print(f"❌ Shop by mark endpoint failed: {response.status_code}")
            print(f"   Response: {response.text}")
            return False
    except Exception as e:
        print(f"❌ Shop by mark endpoint error: {str(e)}")
        return False

def test_analyze_structure_endpoint():
    """Test analyze structure endpoint"""
    try:
        params = {"url": "https://febest.com.au/search?q=brake"}
        response = requests.get(f"{BASE_URL}/analyze-structure", params=params, timeout=30)
        if response.status_code == 200:
            data = response.json()
            print("✅ Analyze structure endpoint: OK")
            print(f"   Found {len(data.get('analysis', {}).get('product_containers', []))} product containers")
            return True
        else:
            print(f"❌ Analyze structure endpoint failed: {response.status_code}")
            print(f"   Response: {response.text}")
            return False
    except Exception as e:
        print(f"❌ Analyze structure endpoint error: {str(e)}")
        return False

def main():
    """Run all API tests"""
    print("🔍 Testing API endpoints...\n")
    
    tests = [
        ("Health Check", test_health_endpoint),
        ("Search", test_search_endpoint),
        ("Shop by Mark", test_shop_by_mark_endpoint),
        ("Analyze Structure", test_analyze_structure_endpoint)
    ]
    
    results = []
    for test_name, test_func in tests:
        print(f"Testing {test_name}...")
        result = test_func()
        results.append(result)
        print()
    
    passed = sum(results)
    total = len(results)
    
    print(f"📊 Test Results: {passed}/{total} tests passed")
    
    if passed == total:
        print("🎉 All API tests passed!")
    else:
        print("⚠️  Some API tests failed. Check the server logs for details.")

if __name__ == "__main__":
    main()