#!/usr/bin/env python3
"""
Quick debug script for category issues
"""

import requests
from scraper_utils import parse_products_with_browser
from browser_utils import create_browser
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from bs4 import BeautifulSoup
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def test_category_directly():
    """Test category URL directly with browser"""
    print("=== Testing Subaru Category Directly ===")
    
    url = "https://febest.com.au/subaru.html"
    print(f"Testing URL: {url}")
    
    browser = None
    try:
        browser = create_browser()
        if not browser:
            print("❌ Failed to create browser")
            return False
            
        print("✅ Browser created successfully")
        browser.get(url)
        
        # Wait for page to load
        WebDriverWait(browser, 15).until(
            EC.presence_of_element_located((By.TAG_NAME, "body"))
        )
        
        print("✅ Page loaded")
        
        # Check if page exists
        soup = BeautifulSoup(browser.page_source, 'html.parser')
        
        # Check for 404 or error messages
        error_indicators = [
            "404", "not found", "page not found", 
            "error", "whoops", "sorry"
        ]
        
        page_text = soup.get_text().lower()
        for indicator in error_indicators:
            if indicator in page_text:
                print(f"❌ Found error indicator: {indicator}")
                return False
        
        # Look for product containers
        product_selectors = [
            '.product-item',
            '.product',
            '.item',
            '[class*="product"]',
            '.category-products',
            '.products'
        ]
        
        found_products = False
        for selector in product_selectors:
            elements = soup.select(selector)
            if elements:
                print(f"✅ Found {len(elements)} elements with selector: {selector}")
                found_products = True
                break
        
        if not found_products:
            print("❌ No product containers found")
            # Save page for debugging
            with open('debug_subaru_page.html', 'w', encoding='utf-8') as f:
                f.write(browser.page_source)
            print("💾 Saved page content to debug_subaru_page.html")
        
        return found_products
        
    except Exception as e:
        print(f"❌ Error: {e}")
        return False
    finally:
        if browser:
            browser.quit()

def test_alternative_urls():
    """Test alternative category URLs"""
    print("\n=== Testing Alternative URLs ===")
    
    urls_to_test = [
        "https://febest.com.au/subaru.html",
        "https://febest.com.au/subaru",
        "https://febest.com.au/catalog/subaru",
        "https://febest.com.au/category/subaru",
        "https://febest.com.au/brand/subaru"
    ]
    
    for url in urls_to_test:
        print(f"\nTesting: {url}")
        try:
            response = requests.get(url, timeout=10)
            print(f"Status: {response.status_code}")
            
            if response.status_code == 200:
                if "product" in response.text.lower():
                    print("✅ Contains product references")
                else:
                    print("⚠️ No product references found")
            elif response.status_code == 404:
                print("❌ Page not found")
            else:
                print(f"⚠️ Unexpected status: {response.status_code}")
                
        except Exception as e:
            print(f"❌ Error: {e}")

def test_api_endpoint():
    """Test API endpoint"""
    print("\n=== Testing API Endpoint ===")
    
    try:
        response = requests.get("http://localhost:8000/shop-by-mark?mark=subaru", timeout=30)
        print(f"Status: {response.status_code}")
        
        if response.status_code == 200:
            data = response.json()
            print(f"Success: {data.get('success')}")
            print(f"Total products: {data.get('total_products')}")
            
            if data.get('products'):
                print("✅ API returned products")
                for i, product in enumerate(data['products'][:2]):
                    print(f"  {i+1}. {product.get('name', 'No name')}")
            else:
                print("❌ API returned no products")
        else:
            print(f"❌ API error: {response.text}")
            
    except Exception as e:
        print(f"❌ API test error: {e}")

def main():
    """Main test function"""
    print("🔍 Debugging Subaru category issue...")
    
    # Test 1: Direct browser test
    browser_success = test_category_directly()
    
    # Test 2: Alternative URLs
    test_alternative_urls()
    
    # Test 3: API endpoint
    test_api_endpoint()
    
    print(f"\n=== SUMMARY ===")
    print(f"Direct browser test: {'PASS' if browser_success else 'FAIL'}")
    
    if not browser_success:
        print("\n💡 Suggestions:")
        print("1. Check if subaru.html page exists on febest.com.au")
        print("2. Try different category URL formats")
        print("3. Check the saved debug_subaru_page.html file")

if __name__ == "__main__":
    main()