#!/usr/bin/env python3
"""
Test script for corrected URL formats based on real febest.com.au structure
"""

import requests
import time
from urllib.parse import quote

BASE_URL = "https://febest.com.au"

def test_url_accessibility(url, description):
    """Test if URL is accessible and returns valid content"""
    print(f"\n🔍 Testing {description}")
    print(f"URL: {url}")
    
    try:
        response = requests.get(url, timeout=10, headers={
            'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36'
        })
        
        print(f"Status Code: {response.status_code}")
        print(f"Content Length: {len(response.content)} bytes")
        
        if response.status_code == 200:
            content = response.text.lower()
            
            # Check for product indicators
            product_indicators = [
                'product-item', 'product-name', 'add to cart', 
                'price', 'catalog', 'search results', 'products'
            ]
            
            found_indicators = [indicator for indicator in product_indicators if indicator in content]
            print(f"Product indicators found: {found_indicators}")
            
            # Check for error indicators
            error_indicators = ['404', 'not found', 'error', 'page not found']
            found_errors = [error for error in error_indicators if error in content]
            
            if found_errors:
                print(f"❌ Error indicators found: {found_errors}")
                return False
            elif found_indicators:
                print(f"✅ Success - Found product indicators")
                return True
            else:
                print(f"⚠️ Page loads but no clear product indicators")
                return False
        else:
            print(f"❌ HTTP Error: {response.status_code}")
            return False
            
    except Exception as e:
        print(f"❌ Request failed: {str(e)}")
        return False

def main():
    """Test all corrected URL formats"""
    print("🚀 Testing Corrected URL Formats for febest.com.au")
    print("=" * 60)
    
    # Test marks to try
    test_marks = ['nissan', 'toyota', 'subaru', 'honda', 'mazda']
    
    results = {}
    
    for mark in test_marks:
        print(f"\n📋 Testing mark: {mark.upper()}")
        print("-" * 40)
        
        mark_results = {}
        
        # Test category URLs (corrected format)
        category_urls = [
            f"{BASE_URL}/shop-parts.html?find={mark.lower()}",
            f"{BASE_URL}/shop-parts.html?find={mark}",
            f"{BASE_URL}/shop-parts.html?find={mark.upper()}"
        ]
        
        for i, url in enumerate(category_urls, 1):
            success = test_url_accessibility(url, f"Category URL {i} for {mark}")
            mark_results[f"category_{i}"] = success
            time.sleep(1)  # Be respectful to the server
        
        # Test search URL (corrected format)
        search_url = f"{BASE_URL}/catalogsearch/result/index/?q={quote(mark)}"
        search_success = test_url_accessibility(search_url, f"Search for {mark}")
        mark_results["search"] = search_success
        
        results[mark] = mark_results
        time.sleep(2)  # Pause between marks
    
    # Test specific product URL
    print(f"\n📦 Testing Product URL")
    print("-" * 40)
    product_url = "https://febest.com.au/0133-001-febest-rear-leaf-spring-u-bolt-repairkit.html"
    product_success = test_url_accessibility(product_url, "Known product URL")
    
    # Summary
    print(f"\n📊 SUMMARY")
    print("=" * 60)
    
    for mark, mark_results in results.items():
        print(f"\n{mark.upper()}:")
        for test_type, success in mark_results.items():
            status = "✅ PASS" if success else "❌ FAIL"
            print(f"  {test_type}: {status}")
    
    print(f"\nProduct URL: {'✅ PASS' if product_success else '❌ FAIL'}")
    
    # Recommendations
    print(f"\n💡 RECOMMENDATIONS")
    print("-" * 40)
    
    # Find working patterns
    working_patterns = []
    for mark, mark_results in results.items():
        for test_type, success in mark_results.items():
            if success:
                working_patterns.append(f"{mark} - {test_type}")
    
    if working_patterns:
        print("✅ Working patterns found:")
        for pattern in working_patterns:
            print(f"  - {pattern}")
    else:
        print("❌ No working patterns found - may need further investigation")
    
    print(f"\n🔧 Next steps:")
    print("1. Run the API server: python main.py")
    print("2. Test the corrected endpoints:")
    print("   curl 'http://localhost:8000/shop-by-mark?mark=nissan'")
    print("   curl 'http://localhost:8000/search?query=nissan'")

if __name__ == "__main__":
    main()