#!/usr/bin/env python3
"""
Test script to compare results for different car marks
"""
import requests
import json
import time

def test_mark_differences():
    """Test if different marks return different results"""
    base_url = "http://localhost:8000"
    marks = ['nissan', 'toyota', 'subaru', 'honda', 'mazda']
    
    print("🔍 Testing different car marks...")
    print("=" * 60)
    
    results = {}
    
    for mark in marks:
        print(f"\n📋 Testing mark: {mark.upper()}")
        try:
            response = requests.get(f"{base_url}/shop-by-mark", 
                                  params={"mark": mark}, 
                                  timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                total_products = data.get('total_products', 0)
                products = data.get('products', [])
                
                # Get first few product names for comparison
                product_names = [p.get('name', 'No name') for p in products[:3]]
                
                results[mark] = {
                    'total_products': total_products,
                    'first_products': product_names,
                    'url_used': data.get('url_used', 'Unknown'),
                    'status': 'success'
                }
                
                print(f"   ✅ Total products: {total_products}")
                print(f"   🔗 URL used: {data.get('url_used', 'Unknown')}")
                print(f"   📦 First 3 products: {product_names}")
                
            else:
                results[mark] = {
                    'status': 'error',
                    'status_code': response.status_code,
                    'error': response.text[:200]
                }
                print(f"   ❌ Error: {response.status_code}")
                
        except Exception as e:
            results[mark] = {
                'status': 'exception',
                'error': str(e)
            }
            print(f"   💥 Exception: {e}")
        
        time.sleep(1)  # Small delay between requests
    
    # Compare results
    print("\n" + "=" * 60)
    print("📊 COMPARISON ANALYSIS")
    print("=" * 60)
    
    # Check if all successful results are identical
    successful_results = {k: v for k, v in results.items() if v.get('status') == 'success'}
    
    if len(successful_results) < 2:
        print("❌ Not enough successful results to compare")
        return
    
    # Compare total products
    total_products_list = [v['total_products'] for v in successful_results.values()]
    if len(set(total_products_list)) == 1:
        print(f"⚠️  ALL MARKS RETURN SAME NUMBER OF PRODUCTS: {total_products_list[0]}")
    else:
        print("✅ Different marks return different number of products")
        for mark, result in successful_results.items():
            print(f"   {mark}: {result['total_products']} products")
    
    # Compare URLs used
    urls_used = [v['url_used'] for v in successful_results.values()]
    if len(set(urls_used)) == 1:
        print(f"⚠️  ALL MARKS USE SAME URL: {urls_used[0]}")
    else:
        print("✅ Different marks use different URLs")
        for mark, result in successful_results.items():
            print(f"   {mark}: {result['url_used']}")
    
    # Compare first products
    print("\n🔍 First products comparison:")
    for mark, result in successful_results.items():
        print(f"   {mark}: {result['first_products']}")
    
    # Check if first products are identical
    first_products_sets = [set(v['first_products']) for v in successful_results.values()]
    if len(first_products_sets) > 1:
        # Check if any two sets are identical
        identical_found = False
        marks_list = list(successful_results.keys())
        for i in range(len(marks_list)):
            for j in range(i + 1, len(marks_list)):
                mark1, mark2 = marks_list[i], marks_list[j]
                if set(successful_results[mark1]['first_products']) == set(successful_results[mark2]['first_products']):
                    print(f"⚠️  {mark1.upper()} and {mark2.upper()} return IDENTICAL products!")
                    identical_found = True
        
        if not identical_found:
            print("✅ All marks return different products")
    
    # Save detailed results
    with open('mark_comparison_results.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print(f"\n💾 Detailed results saved to mark_comparison_results.json")
    
    return results

def test_direct_urls():
    """Test direct URLs to see what the site actually returns"""
    print("\n" + "=" * 60)
    print("🌐 TESTING DIRECT SITE URLs")
    print("=" * 60)
    
    test_urls = [
        ("https://febest.com.au/shop-parts.html?find=nissan", "Nissan"),
        ("https://febest.com.au/shop-parts.html?find=toyota", "Toyota"),
        ("https://febest.com.au/shop-parts.html?find=subaru", "Subaru"),
    ]
    
    for url, mark in test_urls:
        print(f"\n🔗 Testing {mark}: {url}")
        try:
            response = requests.get(url, timeout=30)
            print(f"   Status: {response.status_code}")
            
            if response.status_code == 200:
                content = response.text
                # Look for product indicators
                product_indicators = [
                    'product-item',
                    'product-name',
                    'price',
                    'add-to-cart',
                    'product-list',
                    'catalog-product'
                ]
                
                found_indicators = []
                for indicator in product_indicators:
                    if indicator in content.lower():
                        found_indicators.append(indicator)
                
                print(f"   Product indicators found: {found_indicators}")
                
                # Check if mark is mentioned in content
                mark_mentions = content.lower().count(mark.lower())
                print(f"   '{mark}' mentioned {mark_mentions} times in content")
                
                # Look for "no results" or similar messages
                no_results_indicators = [
                    'no products',
                    'no results',
                    'not found',
                    'no items',
                    'empty'
                ]
                
                found_no_results = []
                for indicator in no_results_indicators:
                    if indicator in content.lower():
                        found_no_results.append(indicator)
                
                if found_no_results:
                    print(f"   ⚠️  'No results' indicators: {found_no_results}")
                
        except Exception as e:
            print(f"   💥 Error: {e}")

if __name__ == "__main__":
    print("🚗 Car Mark Comparison Test")
    print("=" * 60)
    
    # Test API endpoints
    api_results = test_mark_differences()
    
    # Test direct URLs
    test_direct_urls()
    
    print("\n" + "=" * 60)
    print("✅ Test completed!")
    print("Check mark_comparison_results.json for detailed results")