#!/usr/bin/env python3
"""
Quick test script to verify the API fixes
"""

import requests
import json
import time

BASE_URL = "http://localhost:8000"

def test_health():
    """Test health endpoint"""
    print("=== Testing Health Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/health")
        print(f"Status: {response.status_code}")
        print(f"Response: {response.json()}")
        return response.status_code == 200
    except Exception as e:
        print(f"Error: {e}")
        return False

def test_search():
    """Test search endpoint"""
    print("\n=== Testing Search Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/search", params={"query": "brake pad"})
        print(f"Status: {response.status_code}")
        data = response.json()
        print(f"Success: {data.get('success')}")
        print(f"Query: {data.get('query')}")
        print(f"Total products: {data.get('total_products')}")
        
        if data.get('products'):
            print("Sample products:")
            for i, product in enumerate(data['products'][:3]):
                print(f"  {i+1}. {product.get('name', 'No name')}")
                print(f"     URL: {product.get('url', 'No URL')}")
                print(f"     Price: {product.get('price', 'No price')}")
        
        return data.get('total_products', 0) > 0
    except Exception as e:
        print(f"Error: {e}")
        return False

def test_shop_by_mark():
    """Test shop-by-mark endpoint"""
    print("\n=== Testing Shop-by-Mark Endpoint ===")
    try:
        response = requests.get(f"{BASE_URL}/shop-by-mark", params={"mark": "subaru"})
        print(f"Status: {response.status_code}")
        data = response.json()
        print(f"Success: {data.get('success')}")
        print(f"Mark: {data.get('mark')}")
        print(f"Total products: {data.get('total_products')}")
        
        if data.get('products'):
            print("Sample products:")
            for i, product in enumerate(data['products'][:3]):
                print(f"  {i+1}. {product.get('name', 'No name')}")
                print(f"     URL: {product.get('url', 'No URL')}")
                print(f"     Price: {product.get('price', 'No price')}")
        
        return data.get('total_products', 0) > 0
    except Exception as e:
        print(f"Error: {e}")
        return False

def main():
    """Main test function"""
    print("Starting API tests...")
    print("Make sure the server is running on http://localhost:8000")
    
    # Wait a bit for server to start
    time.sleep(2)
    
    results = {
        'health': test_health(),
        'search': test_search(),
        'shop_by_mark': test_shop_by_mark()
    }
    
    print("\n=== TEST RESULTS ===")
    for test_name, result in results.items():
        status = "PASS" if result else "FAIL"
        print(f"{test_name}: {status}")
    
    all_passed = all(results.values())
    print(f"\nOverall: {'ALL TESTS PASSED' if all_passed else 'SOME TESTS FAILED'}")
    
    if not all_passed:
        print("\nIf tests failed, check:")
        print("1. Server is running on http://localhost:8000")
        print("2. No firewall blocking the connection")
        print("3. Check server logs for errors")

if __name__ == "__main__":
    main()